// Default icon configurations (used as fallback)
const defaultIconConfigs = {
    left: [
        { name: 'Gmail', url: 'https://mail.google.com', icon: '📧' },
        { name: 'Calendar', url: 'https://calendar.google.com', icon: '📅' },
        { name: 'Drive', url: 'https://drive.google.com', icon: '📁' },
        { name: 'Docs', url: 'https://docs.google.com', icon: '📄' },
        { name: 'Sheets', url: 'https://sheets.google.com', icon: '📊' },
        { name: 'Keep', url: 'https://keep.google.com', icon: '📝' },
        { name: 'News', url: 'https://news.google.com', icon: '📰' },
        { name: 'Gemini', url: 'https://gemini.google.com', icon: '🤖' }
    ],
    right: [
        { name: 'YouTube', url: 'https://www.youtube.com', icon: '▶️' },
        { name: 'YouTube Music', url: 'https://music.youtube.com', icon: '🎵' },
        { name: 'Twitch', url: 'https://www.twitch.tv', icon: '🎮' },
        { name: 'Beacon', url: 'https://beacon.tv/', icon: '📡' },
        { name: 'Dropout', url: 'https://www.dropout.tv/', icon: '🎭' },
        { name: 'JellyFin', url: 'https://vaggumon.me/jellyfin/web/index.html#!/home.html', icon: '🎬' }
    ],
    googleApps: [
        { name: 'Google Maps', url: 'https://maps.google.com', icon: '🗺️' },
        { name: 'Google Photos', url: 'https://photos.google.com', icon: '🖼️' },
        { name: 'Google Dashboard', url: 'https://myaccount.google.com/dashboard', icon: '🏠' },
        { name: 'Google Account', url: 'https://myaccount.google.com', icon: '🔑' },
        { name: 'My Chart', url: 'https://mychart.uwhealth.org/', icon: '🏥' },
        { name: 'Asus Router', url: 'http://router.asus.com/Main_Login.asp', icon: '🌐' },
        { name: 'Blue SKy', url: 'https://bsky.app/', icon: '☁️' }
    ]
};

// Current icon configurations (can be customized)
// Start with empty arrays for first-time load (user will set up icons via setup wizard)
let iconConfigs = {
    left: [],
    right: [],
    googleApps: []
};

// Available icon files in the Icons folder
const availableIcons = [
    'gmail.png',
    'google-calendar.png',
    'google-drive.png',
    'google-docs.png',
    'google-sheets.png',
    'google-keep.png',
    'google-news.png',
    'google-gemini.png',
    'youtube.png',
    'youtubemusic.png',
    '939743_twitch_twitch.tv icon_icon.png',
    'beacontv.png',
    'dropouttv.png',
    'jellyfin.png',
    'google-maps.png',
    'google-photos.png',
    'google-dashboard.png',
    'google-account.png',
    'mychart.png',
    'asusrouter.png',
    'Bluesky.png'
];

// CoreUI Icons (from https://github.com/coreui/coreui-icons)
// These are SVG icons available via raw GitHub URLs
const coreUIIcons = [
    { name: 'Home', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-home.svg' },
    { name: 'Settings', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-settings.svg' },
    { name: 'User', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-user.svg' },
    { name: 'Mail', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-envelope.svg' },
    { name: 'Calendar', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-calendar.svg' },
    { name: 'Folder', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-folder.svg' },
    { name: 'File', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-file.svg' },
    { name: 'Search', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-magnifying-glass.svg' },
    { name: 'Star', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-star.svg' },
    { name: 'Heart', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-heart.svg' },
    { name: 'Bell', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-bell.svg' },
    { name: 'Lock', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-lock-locked.svg' },
    { name: 'Unlock', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-lock-unlocked.svg' },
    { name: 'Play', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-media-play.svg' },
    { name: 'Pause', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-media-pause.svg' },
    { name: 'Music', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-music-note.svg' },
    { name: 'Video', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-video.svg' },
    { name: 'Image', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-image.svg' },
    { name: 'Cloud', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-cloud-upload.svg' },
    { name: 'Download', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-cloud-download.svg' },
    { name: 'Share', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-share.svg' },
    { name: 'Link', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-link.svg' },
    { name: 'Trash', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-trash.svg' },
    { name: 'Edit', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-pencil.svg' },
    { name: 'Plus', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-plus.svg' },
    { name: 'Minus', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-minus.svg' },
    { name: 'Check', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-check.svg' },
    { name: 'X (Close)', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-x.svg' },
    { name: 'Menu', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-menu.svg' },
    { name: 'Apps', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-apps.svg' },
    { name: 'Grid', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-grid.svg' },
    { name: 'List', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-list.svg' },
    { name: 'Info', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-info.svg' },
    { name: 'Warning', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-warning.svg' },
    { name: 'Shield', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-shield-alt.svg' },
    { name: 'Globe', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-globe-alt.svg' },
    { name: 'Phone', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-phone.svg' },
    { name: 'Chat', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-chat-bubble.svg' },
    { name: 'Bookmark', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-bookmark.svg' },
    { name: 'Tag', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-tag.svg' },
    { name: 'Filter', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-filter.svg' },
    { name: 'Sort', url: 'https://raw.githubusercontent.com/coreui/coreui-icons/main/svg/free/cil-sort-ascending.svg' }
];

// Map service names to icon filenames
function getIconFilename(serviceName) {
    const iconMap = {
        'Gmail': 'gmail.png',
        'Calendar': 'google-calendar.png',
        'Drive': 'google-drive.png',
        'Docs': 'google-docs.png',
        'Sheets': 'google-sheets.png',
        'Keep': 'google-keep.png',
        'News': 'google-news.png',
        'Gemini': 'google-gemini.png',
        'YouTube': 'youtube.png',
        'YouTube Music': 'youtubemusic.png',
        'Twitch': '939743_twitch_twitch.tv icon_icon.png',
        'Beacon': 'beacontv.png',
        'Dropout': 'dropouttv.png',
        'JellyFin': 'jellyfin.png',
        'Google Maps': 'google-maps.png',
        'Google Photos': 'google-photos.png',
        'Google Dashboard': 'google-dashboard.png',
        'Google Account': 'google-account.png',
        'My Chart': 'mychart.png',
        'Asus Router': 'asusrouter.png',
        'Blue SKy': 'Bluesky.png'
    };
    return iconMap[serviceName] || null;
}

// Panel management
let panelData = [];
let columnCount = 3;
let panelIdCounter = 0;
let bookmarkIdCounter = 0;
let currentTheme = 'glass';
let backgroundImageData = null;
let editingBookmarkId = null;
let hideAddBookmarkButtons = false;

// Background images for glass theme (random selection)
// Supports JPG, PNG, and WEBP formats
// User can add their own images to the backgrounds folder and update this array
// A random image will be selected each time the glass theme loads
const backgroundImages = [
    'Background0.jpg',
    'Background01.jpg',
    'Background02.jpg',
    'Background03.jpg',
    'Background04.jpg',
    'Background05.jpg',
    'Background06.jpg',
    'Background08.jpg',
    'Background09.jpg',
    'Background10.jpg',
    'Background11.jpg',
    'Background12.jpg',
    'Background13.jpg',
    'Background14.jpg',
    'Background15.jpg',
    'Background16.jpg',
    'Background17.jpg',
    'Background18.jpg',
    'Background19.jpg',
    'Background20.jpg'
];

// Panel border colors for colorful theme
const panelColors = [
    '#920194',
    '#168C02',
    '#02B3AD',
    '#4404B3',
    '#965108',
    '#800602',
    '#0D0385',
    '#8A0459',
    '#8F8A06',
    '#7602A8'
];

// Get a random color from the panel colors array
function getRandomPanelColor() {
    return panelColors[Math.floor(Math.random() * panelColors.length)];
}

// Convert hex to RGB for rgba calculations
function hexToRgb(hex) {
    const result = /^#?([a-f\d]{2})([a-f\d]{2})([a-f\d]{2})$/i.exec(hex);
    return result ? {
        r: parseInt(result[1], 16),
        g: parseInt(result[2], 16),
        b: parseInt(result[3], 16)
    } : null;
}
let frostedEffectLevel = 15; // Default blur level in pixels
let panelDarknessLevel = 0; // Default panel darkness (0-80%)

// Drag and drop state
let draggedBookmark = null;
let draggedPanelId = null;

// Reset all data for deployment/first-time setup
function resetToDefaults() {
    try {
        // Clear all localStorage
        localStorage.removeItem('browserHomepageData');
        localStorage.removeItem('browserHomepageSetupCompleted');
        
        // Reset all variables to defaults
        panelData = [];
        columnCount = 3;
        panelIdCounter = 0;
        bookmarkIdCounter = 0;
        currentTheme = 'glass';
        backgroundImageData = null;
        editingBookmarkId = null;
        hideAddBookmarkButtons = false;
        frostedEffectLevel = 15;
        panelDarknessLevel = 0;
        
        // Clear icon configs (empty arrays, not default icons)
        iconConfigs = {
            left: [],
            right: [],
            googleApps: []
        };
        
        // Reload the page to apply defaults
        window.location.reload();
    } catch (error) {
        alert('Error resetting data: ' + error.message);
    }
}

// Make resetToDefaults globally accessible
window.resetToDefaults = resetToDefaults;

// Global reset function for reset button
window.resetAllData = function() {
    if (confirm('⚠️ WARNING: This will reset ALL data to defaults.\n\n' +
               'This includes:\n' +
               '- All panels and bookmarks\n' +
               '- Icon customizations\n' +
               '- Theme settings\n' +
               '- Background images\n' +
               '- All other preferences\n\n' +
               'This action cannot be undone!\n\n' +
               'Are you sure you want to continue?')) {
        try {
            // Save empty/default state to localStorage before clearing
            const resetData = {
                theme: 'glass',
                columnCount: 3,
                panels: [],
                panelIdCounter: 0,
                bookmarkIdCounter: 0,
                hideAddBookmarkButtons: false,
                frostedEffectLevel: 15,
                panelDarknessLevel: 0,
                iconConfigs: {
                    left: [],
                    right: [],
                    googleApps: []
                },
                version: '1.0'
            };
            
            // Save reset data
            localStorage.setItem('browserHomepageData', JSON.stringify(resetData));
            // Clear setup completed flag so setup wizard can show again if needed
            localStorage.removeItem('browserHomepageSetupCompleted');
            
            // Reload page
            window.location.reload();
        } catch (e) {
            alert('Error: ' + e.message);
            // Fallback: clear everything and reload
            localStorage.clear();
            window.location.reload();
        }
    }
};

// For deployment: Uncomment the line below to reset all data
// resetToDefaults();

// Initialize the application
document.addEventListener('DOMContentLoaded', function() {
    loadData();
    initializeIcons();
    initializePanels();
    initializeTheme();
    initializeBookmarkModal();
    initializeSettings();
    initializeSetupWizard();
    initializeTutorialWizard();
    initializeAboutModal();
});

// Initialize icon navigation bars
function initializeIcons() {
    const leftGroup = document.querySelector('.icon-group-left');
    const rightGroup = document.querySelector('.icon-group-right');
    const settingsBtn = document.getElementById('settingsToggleBtn');

    // Create left side icons
    iconConfigs.left.forEach(icon => {
        const iconElement = createIconElement(icon);
        leftGroup.appendChild(iconElement);
    });

    // Create Other Apps dropdown
    const dropdown = createDropdown('▼', iconConfigs.googleApps);
    leftGroup.appendChild(dropdown);

    // Create right side icons
    iconConfigs.right.forEach(icon => {
        const iconElement = createIconElement(icon);
        rightGroup.appendChild(iconElement);
    });

    // Move settings button to the very end
    if (settingsBtn && settingsBtn.parentNode === rightGroup) {
        rightGroup.removeChild(settingsBtn);
    }
    rightGroup.appendChild(settingsBtn);
}

// Create an icon element
function createIconElement(config) {
    const iconLink = document.createElement('a');
    iconLink.href = config.url;
    iconLink.target = '_blank';
    iconLink.rel = 'noopener noreferrer';
    iconLink.className = 'icon-link';
    iconLink.setAttribute('title', config.name);
    iconLink.setAttribute('aria-label', config.name);

    // Priority: customIcon > iconFilename > getIconFilename(name) > emoji
    if (config.customIcon) {
        // Use custom uploaded icon
        const iconImg = document.createElement('img');
        iconImg.src = config.customIcon;
        iconImg.alt = config.name;
        iconImg.className = 'icon-image';
        iconImg.onerror = function() {
            // Fallback to emoji if custom icon fails to load
            const iconSpan = document.createElement('span');
            iconSpan.className = 'icon-emoji';
            iconSpan.textContent = config.icon || '🔗';
            iconLink.innerHTML = '';
            iconLink.appendChild(iconSpan);
        };
        iconLink.appendChild(iconImg);
    } else if (config.iconFilename || config.iconUrl) {
        // Use selected icon - check if it's a URL (CoreUI) or local file
        const iconSrc = config.iconFilename || config.iconUrl;
        const isUrl = iconSrc.startsWith('http://') || iconSrc.startsWith('https://');
        const iconImg = document.createElement('img');
        iconImg.src = isUrl ? iconSrc : `Icons/${iconSrc}`;
        iconImg.alt = config.name;
        iconImg.className = 'icon-image';
        iconImg.onerror = function() {
            // Fallback to emoji if image fails to load
            const iconSpan = document.createElement('span');
            iconSpan.className = 'icon-emoji';
            iconSpan.textContent = config.icon || '🔗';
            iconLink.innerHTML = '';
            iconLink.appendChild(iconSpan);
        };
        iconLink.appendChild(iconImg);
    } else {
        // Check if we have an icon file for this service (auto-detect)
        const iconFilename = getIconFilename(config.name);
        if (iconFilename) {
            // Use image file
            const iconImg = document.createElement('img');
            iconImg.src = `Icons/${iconFilename}`;
            iconImg.alt = config.name;
            iconImg.className = 'icon-image';
            iconImg.onerror = function() {
                // Fallback to emoji if image fails to load
                const iconSpan = document.createElement('span');
                iconSpan.className = 'icon-emoji';
                iconSpan.textContent = config.icon || '🔗';
                iconLink.innerHTML = '';
                iconLink.appendChild(iconSpan);
            };
            iconLink.appendChild(iconImg);
        } else {
            // Fallback to emoji
            const iconSpan = document.createElement('span');
            iconSpan.className = 'icon-emoji';
            iconSpan.textContent = config.icon || '🔗';
            iconLink.appendChild(iconSpan);
        }
    }

    return iconLink;
}

// Create dropdown for Other Apps
function createDropdown(label, apps) {
    const dropdownContainer = document.createElement('div');
    dropdownContainer.className = 'dropdown-container';

    const dropdownButton = document.createElement('button');
    dropdownButton.className = 'dropdown-button';
    dropdownButton.textContent = label;
    dropdownButton.setAttribute('aria-label', 'More Other Apps');
    dropdownButton.setAttribute('aria-haspopup', 'true');
    dropdownButton.setAttribute('aria-expanded', 'false');

    const dropdownMenu = document.createElement('div');
    dropdownMenu.className = 'dropdown-menu';
    dropdownMenu.setAttribute('role', 'menu');

    apps.forEach(app => {
        const menuItem = document.createElement('a');
        menuItem.href = app.url;
        menuItem.target = '_blank';
        menuItem.rel = 'noopener noreferrer';
        menuItem.className = 'dropdown-item';
        menuItem.setAttribute('role', 'menuitem');
        
        // Priority: customIcon > iconFilename > getIconFilename(name) > emoji
        if (app.customIcon) {
            // Use custom uploaded icon
            const iconImg = document.createElement('img');
            iconImg.src = app.customIcon;
            iconImg.alt = app.name;
            iconImg.className = 'dropdown-icon-image';
            iconImg.onerror = function() {
                // Fallback to emoji if custom icon fails to load
                const iconSpan = document.createElement('span');
                iconSpan.className = 'icon-emoji';
                iconSpan.textContent = app.icon || '🔗';
                menuItem.insertBefore(iconSpan, menuItem.firstChild);
                menuItem.removeChild(iconImg);
            };
            menuItem.appendChild(iconImg);
        } else if (app.iconFilename) {
            // Use selected icon - check if it's a URL (CoreUI) or local file
            const iconSrc = app.iconFilename;
            const isUrl = iconSrc.startsWith('http://') || iconSrc.startsWith('https://');
            const iconImg = document.createElement('img');
            iconImg.src = isUrl ? iconSrc : `Icons/${iconSrc}`;
            iconImg.alt = app.name;
            iconImg.className = 'dropdown-icon-image';
            iconImg.onerror = function() {
                // Fallback to emoji if image fails to load
                const iconSpan = document.createElement('span');
                iconSpan.className = 'icon-emoji';
                iconSpan.textContent = app.icon || '🔗';
                menuItem.insertBefore(iconSpan, menuItem.firstChild);
                menuItem.removeChild(iconImg);
            };
            menuItem.appendChild(iconImg);
        } else {
            // Check if we have an icon file for this service (auto-detect)
            const iconFilename = getIconFilename(app.name);
            if (iconFilename) {
                // Use image file
                const iconImg = document.createElement('img');
                iconImg.src = `Icons/${iconFilename}`;
                iconImg.alt = app.name;
                iconImg.className = 'dropdown-icon-image';
                iconImg.onerror = function() {
                    // Fallback to emoji if image fails to load
                    const iconSpan = document.createElement('span');
                    iconSpan.className = 'icon-emoji';
                    iconSpan.textContent = app.icon || '🔗';
                    menuItem.insertBefore(iconSpan, menuItem.firstChild);
                    menuItem.removeChild(iconImg);
                };
                menuItem.appendChild(iconImg);
            } else {
                // Fallback to emoji
                const iconSpan = document.createElement('span');
                iconSpan.className = 'icon-emoji';
                iconSpan.textContent = app.icon || '🔗';
                menuItem.appendChild(iconSpan);
            }
        }
        
        const textSpan = document.createElement('span');
        textSpan.textContent = app.name;

        menuItem.appendChild(textSpan);
        dropdownMenu.appendChild(menuItem);
    });

    dropdownButton.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        const isExpanded = dropdownButton.getAttribute('aria-expanded') === 'true';
        dropdownButton.setAttribute('aria-expanded', !isExpanded);
        dropdownMenu.classList.toggle('show');
    });

    // Close dropdown when clicking outside
    document.addEventListener('click', function(e) {
        if (!dropdownContainer.contains(e.target)) {
            dropdownMenu.classList.remove('show');
            dropdownButton.setAttribute('aria-expanded', 'false');
        }
    });

    dropdownContainer.appendChild(dropdownButton);
    dropdownContainer.appendChild(dropdownMenu);
    return dropdownContainer;
}

// Initialize panel system
function initializePanels() {
    // Set initial column count
    updateColumnLayout(columnCount);

    // Render existing panels
    renderPanels();
}

// Update column layout
function updateColumnLayout(columns) {
    const panelsContainer = document.getElementById('panelsContainer');
    if (!panelsContainer) return;
    panelsContainer.style.gridTemplateColumns = `repeat(${columns}, 1fr)`;
    columnCount = columns;
}

// Add a new panel
function addPanel() {
    const panelId = `panel-${panelIdCounter++}`;
    const newPanel = {
        id: panelId,
        title: 'New Panel',
        bookmarks: []
    };
    panelData.push(newPanel);
    renderPanels();
}

// Remove a panel
function removePanel(panelId) {
    panelData = panelData.filter(panel => panel.id !== panelId);
    renderPanels();
    saveData();
}

// Render all panels
function renderPanels() {
    const panelsContainer = document.getElementById('panelsContainer');
    if (!panelsContainer) return;
    
    panelsContainer.innerHTML = '';

    panelData.forEach(panel => {
        const panelElement = createPanelElement(panel);
        panelsContainer.appendChild(panelElement);
    });
}

// Create panel element
function createPanelElement(panel) {
      const panelDiv = document.createElement('div');
      panelDiv.className = 'panel';
      panelDiv.setAttribute('data-panel-id', panel.id);
  
            // Apply random color for colorful theme
      let panelColor = null;
      if (currentTheme === 'colorful') {
          panelColor = getRandomPanelColor();
          const rgb = hexToRgb(panelColor);
          
          // Apply border color and box shadow
          panelDiv.style.borderColor = panelColor;
          panelDiv.style.boxShadow = `0 8px 32px rgba(0, 0, 0, 0.5), 0 0 20px rgba(${rgb.r}, ${rgb.g}, ${rgb.b}, 0.3)`;
      }

      const panelHeader = document.createElement('div');
      panelHeader.className = 'panel-header';
      
      // Apply header background color for colorful theme (same color as panel)
      if (currentTheme === 'colorful' && panelColor) {
          panelHeader.style.borderBottomColor = panelColor;
          panelHeader.style.background = panelColor;
      }

    const panelTitle = document.createElement('h3');
    panelTitle.className = 'panel-title';
    panelTitle.textContent = panel.title;
    panelTitle.setAttribute('contenteditable', 'false');
    panelTitle.title = 'Click to edit title';
    panelTitle.style.cursor = 'pointer';
    
    // Make title editable on click
    panelTitle.addEventListener('click', function() {
        if (this.getAttribute('contenteditable') === 'false') {
            this.setAttribute('contenteditable', 'true');
            this.focus();
            // Select all text
            const range = document.createRange();
            range.selectNodeContents(this);
            const selection = window.getSelection();
            selection.removeAllRanges();
            selection.addRange(range);
        }
    });
    
    panelTitle.addEventListener('blur', function() {
        this.setAttribute('contenteditable', 'false');
        const newTitle = this.textContent.trim();
        if (newTitle) {
            panel.title = newTitle;
            saveData();
        } else {
            this.textContent = panel.title; // Revert if empty
        }
    });
    
    panelTitle.addEventListener('keydown', function(e) {
        if (e.key === 'Enter') {
            e.preventDefault();
            this.blur();
        }
        if (e.key === 'Escape') {
            this.textContent = panel.title;
            this.blur();
        }
    });

    const panelActions = document.createElement('div');
    panelActions.className = 'panel-actions';

    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'panel-delete-btn';
    deleteBtn.innerHTML = '×';
    deleteBtn.setAttribute('aria-label', 'Delete panel');
    deleteBtn.title = 'Delete Panel';
    deleteBtn.addEventListener('click', function() {
        if (confirm('Are you sure you want to delete this panel?')) {
            removePanel(panel.id);
        }
    });

    panelActions.appendChild(deleteBtn);
    panelHeader.appendChild(panelTitle);
    panelHeader.appendChild(panelActions);

    const panelBody = document.createElement('div');
    panelBody.className = 'panel-body';

    // Add drop zone functionality
    panelDiv.addEventListener('dragover', function(e) {
        e.preventDefault();
        e.dataTransfer.dropEffect = 'move';
        if (draggedBookmark && draggedPanelId !== panel.id) {
            this.classList.add('drop-zone');
        }
    });

    panelDiv.addEventListener('dragleave', function(e) {
        // Only remove drop-zone if we're leaving the panel entirely
        if (!this.contains(e.relatedTarget)) {
            this.classList.remove('drop-zone');
        }
    });

    panelDiv.addEventListener('drop', function(e) {
        e.preventDefault();
        this.classList.remove('drop-zone');
        
        if (draggedBookmark && draggedPanelId && draggedPanelId !== panel.id) {
            // Remove bookmark from source panel
            const sourcePanel = panelData.find(p => p.id === draggedPanelId);
            if (sourcePanel) {
                sourcePanel.bookmarks = sourcePanel.bookmarks.filter(b => b.id !== draggedBookmark.id);
            }
            
            // Add bookmark to target panel
            if (!panel.bookmarks) {
                panel.bookmarks = [];
            }
            panel.bookmarks.push(draggedBookmark);
            
            // Re-render panels and save
            renderPanels();
            saveData();
        }
        
        draggedBookmark = null;
        draggedPanelId = null;
    });

    // Add bookmark button (only if not hidden in settings)
    if (!hideAddBookmarkButtons) {
        const addBookmarkBtn = document.createElement('button');
        addBookmarkBtn.className = 'add-bookmark-btn';
        addBookmarkBtn.innerHTML = '+ Add Bookmark';
        addBookmarkBtn.addEventListener('click', function() {
            openBookmarkModal(panel.id);
        });
        panelBody.appendChild(addBookmarkBtn);
    }

    // Bookmarks container
    const bookmarksContainer = document.createElement('div');
    bookmarksContainer.className = 'bookmarks-container';
    bookmarksContainer.id = `bookmarks-${panel.id}`;

    // Render bookmarks
    if (panel.bookmarks && panel.bookmarks.length > 0) {
        panel.bookmarks.forEach(bookmark => {
            const bookmarkElement = createBookmarkElement(bookmark, panel.id);
            bookmarksContainer.appendChild(bookmarkElement);
        });
    } else {
        const emptyMessage = document.createElement('p');
        emptyMessage.className = 'panel-empty';
        if (hideAddBookmarkButtons) {
            emptyMessage.textContent = 'No bookmarks yet. Right-click panel or use settings to add bookmarks.';
        } else {
            emptyMessage.textContent = 'No bookmarks yet. Click "Add Bookmark" to add one.';
        }
        bookmarksContainer.appendChild(emptyMessage);
    }

    panelBody.appendChild(bookmarksContainer);

    panelDiv.appendChild(panelHeader);
    panelDiv.appendChild(panelBody);

    return panelDiv;
}

// Create bookmark element
function createBookmarkElement(bookmark, panelId) {
    const bookmarkDiv = document.createElement('div');
    bookmarkDiv.className = 'bookmark-item';
    bookmarkDiv.setAttribute('data-bookmark-id', bookmark.id);
    bookmarkDiv.setAttribute('draggable', 'true');
    bookmarkDiv.setAttribute('data-panel-id', panelId);

          const bookmarkLink = document.createElement('a');
      bookmarkLink.href = bookmark.url;
      bookmarkLink.target = '_blank';
      bookmarkLink.rel = 'noopener noreferrer';
      bookmarkLink.className = 'bookmark-link';
      // Prevent link navigation during drag
      bookmarkLink.setAttribute('draggable', 'false');

    const faviconImg = document.createElement('img');
    faviconImg.className = 'bookmark-favicon';
    faviconImg.alt = bookmark.title || 'Bookmark';
    
    // Set favicon with error handling
    const setFaviconSrc = () => {
        if (bookmark.favicon) {
            faviconImg.src = bookmark.favicon;
        } else {
            // Try to extract domain for Google's service
            try {
                const domain = extractDomain(bookmark.url);
                // Skip invalid domains to avoid 404s
                if (domain && !domain.match(/^\d+\.\d+\.\d+\.\d+$/) && domain.includes('.')) {
                    faviconImg.src = `https://www.google.com/s2/favicons?domain=${domain}&sz=32`;
                } else {
                    // For IPs/invalid domains, use a default emoji instead
                    faviconImg.style.display = 'none';
                    const emojiSpan = document.createElement('span');
                    emojiSpan.className = 'bookmark-favicon-emoji';
                    emojiSpan.textContent = '🔗';
                    bookmarkLink.insertBefore(emojiSpan, bookmarkLink.firstChild);
                    return;
                }
            } catch (e) {
                // Invalid URL - hide favicon
                faviconImg.style.display = 'none';
                const emojiSpan = document.createElement('span');
                emojiSpan.className = 'bookmark-favicon-emoji';
                emojiSpan.textContent = '🔗';
                bookmarkLink.insertBefore(emojiSpan, bookmarkLink.firstChild);
                return;
            }
        }
    };
    
    setFaviconSrc();
    
    // Handle favicon load errors gracefully
    faviconImg.onerror = function() {
        // Hide broken image and show emoji fallback
        this.style.display = 'none';
        if (!bookmarkLink.querySelector('.bookmark-favicon-emoji')) {
            const emojiSpan = document.createElement('span');
            emojiSpan.className = 'bookmark-favicon-emoji';
            emojiSpan.textContent = '🔗';
            bookmarkLink.insertBefore(emojiSpan, bookmarkLink.firstChild);
        }
    };

    const bookmarkInfo = document.createElement('div');
    bookmarkInfo.className = 'bookmark-info';

    const bookmarkTitle = document.createElement('div');
    bookmarkTitle.className = 'bookmark-title';
    bookmarkTitle.textContent = bookmark.title || 'Untitled';

    const bookmarkDesc = document.createElement('div');
    bookmarkDesc.className = 'bookmark-description';
    bookmarkDesc.textContent = bookmark.description || '';

    bookmarkInfo.appendChild(bookmarkTitle);
    bookmarkInfo.appendChild(bookmarkDesc);
    bookmarkLink.appendChild(faviconImg);
    bookmarkLink.appendChild(bookmarkInfo);

    const bookmarkActions = document.createElement('div');
    bookmarkActions.className = 'bookmark-actions';

    const editBtn = document.createElement('button');
    editBtn.className = 'bookmark-edit-btn';
    editBtn.innerHTML = '✏️';
    editBtn.title = 'Edit';
    editBtn.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        openBookmarkModal(panelId, bookmark.id);
    });

    const deleteBtn = document.createElement('button');
    deleteBtn.className = 'bookmark-delete-btn';
    deleteBtn.innerHTML = '×';
    deleteBtn.title = 'Delete';
    deleteBtn.addEventListener('click', function(e) {
        e.preventDefault();
        e.stopPropagation();
        if (confirm('Delete this bookmark?')) {
            deleteBookmark(panelId, bookmark.id);
        }
    });

          bookmarkActions.appendChild(editBtn);
      bookmarkActions.appendChild(deleteBtn);

      bookmarkDiv.appendChild(bookmarkLink);
      bookmarkDiv.appendChild(bookmarkActions);

      // Add drag event listeners
      bookmarkDiv.addEventListener('dragstart', function(e) {
        draggedBookmark = bookmark;
        draggedPanelId = panelId;
        this.style.opacity = '0.5';
        e.dataTransfer.effectAllowed = 'move';
        e.dataTransfer.setData('text/html', this.innerHTML);
      });

      bookmarkDiv.addEventListener('dragend', function(e) {
        this.style.opacity = '1';
        // Clear drop indicators
        document.querySelectorAll('.panel.drop-zone').forEach(panel => {
          panel.classList.remove('drop-zone');
        });
      });

      // Prevent drag on action buttons
      editBtn.addEventListener('mousedown', function(e) {
        e.stopPropagation();
      });
      deleteBtn.addEventListener('mousedown', function(e) {
        e.stopPropagation();
      });

      return bookmarkDiv;
}

// Initialize theme system
function initializeTheme() {
    // Set initial theme
    setTheme(currentTheme);
}

// Set theme
function setTheme(theme) {
    document.body.className = `theme-${theme}`;
    
    // Manage html element class for colorful theme background
    if (theme === 'colorful') {
        document.documentElement.classList.add('theme-colorful');
        // Hide background image container for colorful theme
        const bgContainer = document.getElementById('backgroundImageContainer');
        if (bgContainer) {
            bgContainer.style.display = 'none';
        }
    } else {
        document.documentElement.classList.remove('theme-colorful');
        // Show background image container for glass theme
        const bgContainer = document.getElementById('backgroundImageContainer');
        if (bgContainer) {
            bgContainer.style.display = 'block';
        }
        // Load background: use manually uploaded one if available, otherwise random
        const bgImage = document.getElementById('backgroundImage');
        if (bgImage) {
            if (backgroundImageData) {
                // User has manually uploaded a background - use it
                bgImage.src = backgroundImageData;
                bgImage.style.display = 'block';
                bgImage.classList.add('loaded');
            } else {
                // No manual background - load a random one from the array
                loadRandomBackgroundImage();
            }
        }
    }
    
    currentTheme = theme;
    
    // Re-render panels to apply correct theme styles (removes inline styles from other theme)
    renderPanels();
    
    // Show/hide glass theme specific settings
    const frostedEffectSetting = document.getElementById('frostedEffectSetting');
    if (frostedEffectSetting) {
        frostedEffectSetting.style.display = theme === 'glass' ? 'block' : 'none';
    }
    
    const panelDarknessSetting = document.getElementById('panelDarknessSetting');
    if (panelDarknessSetting) {
        panelDarknessSetting.style.display = theme === 'glass' ? 'block' : 'none';
    }
    
    // Apply frosted effect level
    applyFrostedEffect();
    // Apply panel darkness level
    applyPanelDarkness();
    saveData();
}

// Apply frosted effect level
function applyFrostedEffect() {
    if (currentTheme === 'glass') {
        // Set CSS variable on :root so it applies to all elements
        document.documentElement.style.setProperty('--glass-blur', `${frostedEffectLevel}px`);
    }
}

// Apply panel darkness level
function applyPanelDarkness() {
    if (currentTheme === 'glass') {
        // Set CSS variable on :root so it applies to all elements
        // Store as a number (0-80) for calc() compatibility
        document.documentElement.style.setProperty('--panel-darkness', panelDarknessLevel.toString());
    }
}

// Load a random background image from the backgrounds folder
function loadRandomBackgroundImage() {
    if (backgroundImages.length === 0) {
        return;
    }

    const bgImage = document.getElementById('backgroundImage');
    if (!bgImage) return;

    const randomIndex = Math.floor(Math.random() * backgroundImages.length);
    const imagePath = `backgrounds/${backgroundImages[randomIndex]}`;

    // Try to load the image
    const img = new Image();
    img.onload = function() {
        bgImage.src = imagePath;
        bgImage.style.display = 'block';
        bgImage.classList.add('loaded');
    };
    img.onerror = function() {
        // If image fails to load, try another random one (avoid infinite loop)
        const fallbackIndex = (randomIndex + 1) % backgroundImages.length;
        if (fallbackIndex !== randomIndex && backgroundImages.length > 1) {
            const fallbackPath = `backgrounds/${backgroundImages[fallbackIndex]}`;
            const fallbackImg = new Image();
            fallbackImg.onload = function() {
                bgImage.src = fallbackPath;
                bgImage.style.display = 'block';
                bgImage.classList.add('loaded');
            };
            fallbackImg.src = fallbackPath;
        }
    };
    img.src = imagePath;
}

// Set background image (used for manual selection via file picker)
function setBackgroundImage(imageUrl) {
    const bgImage = document.getElementById('backgroundImage');
    if (!bgImage) return;
    bgImage.src = imageUrl;
    bgImage.style.display = 'block';
    bgImage.onload = function() {
        bgImage.classList.add('loaded');
    };
    backgroundImageData = imageUrl;
    saveData();
}

// Initialize bookmark modal
function initializeBookmarkModal() {
    const modal = document.getElementById('bookmarkModal');
    const closeBtn = document.getElementById('closeModal');
    const cancelBtn = document.getElementById('cancelBookmark');
    const form = document.getElementById('bookmarkForm');
    const urlInput = document.getElementById('bookmarkUrl');

    closeBtn.addEventListener('click', closeBookmarkModal);
    cancelBtn.addEventListener('click', closeBookmarkModal);

    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeBookmarkModal();
        }
    });

    form.addEventListener('submit', function(e) {
        e.preventDefault();
        saveBookmark();
    });

    // Auto-fetch metadata when URL is entered (only in add mode, not edit mode)
    urlInput.addEventListener('blur', function() {
        if (!editingBookmarkId && this.value.trim()) {
            fetchBookmarkMetadata(this.value.trim());
        }
    });
}

// Open bookmark modal
function openBookmarkModal(panelId, bookmarkId = null) {
    const modal = document.getElementById('bookmarkModal');
    const form = document.getElementById('bookmarkForm');
    const panelSelect = document.getElementById('bookmarkPanel');
    const modalTitle = document.getElementById('modalTitle');

    // Populate panel dropdown
    panelSelect.innerHTML = '<option value="">Select a panel...</option>';
    panelData.forEach(panel => {
        const option = document.createElement('option');
        option.value = panel.id;
        option.textContent = panel.title;
        if (panel.id === panelId) {
            option.selected = true;
        }
        panelSelect.appendChild(option);
    });

    editingBookmarkId = bookmarkId;

    if (bookmarkId) {
        // Edit mode
        modalTitle.textContent = 'Edit Bookmark';
        const panel = panelData.find(p => p.id === panelId);
        const bookmark = panel.bookmarks.find(b => b.id === bookmarkId);
        
        document.getElementById('bookmarkUrl').value = bookmark.url || '';
        document.getElementById('bookmarkTitle').value = bookmark.title || '';
        document.getElementById('bookmarkDescription').value = bookmark.description || '';
        document.getElementById('bookmarkFavicon').value = bookmark.favicon || '';
        document.getElementById('bookmarkPanel').value = panelId;
    } else {
        // Add mode
        modalTitle.textContent = 'Add Bookmark';
        form.reset();
        document.getElementById('bookmarkPanel').value = panelId;
    }

    modal.classList.add('show');
}

// Close bookmark modal
function closeBookmarkModal() {
    const modal = document.getElementById('bookmarkModal');
    modal.classList.remove('show');
    editingBookmarkId = null;
}

// Save bookmark
function saveBookmark() {
    const url = document.getElementById('bookmarkUrl').value.trim();
    const title = document.getElementById('bookmarkTitle').value.trim();
    const description = document.getElementById('bookmarkDescription').value.trim();
    const favicon = document.getElementById('bookmarkFavicon').value.trim();
    const panelId = document.getElementById('bookmarkPanel').value;

    if (!url || !panelId) {
        alert('URL and Panel are required');
        return;
    }

    const panel = panelData.find(p => p.id === panelId);
    if (!panel) {
        alert('Panel not found');
        return;
    }

    if (editingBookmarkId) {
        // Update existing bookmark
        const bookmark = panel.bookmarks.find(b => b.id === editingBookmarkId);
        if (bookmark) {
            bookmark.url = url;
            bookmark.title = title;
            bookmark.description = description;
            bookmark.favicon = favicon;
        }
    } else {
        // Add new bookmark
        const newBookmark = {
            id: `bookmark-${bookmarkIdCounter++}`,
            url: url,
            title: title,
            description: description,
            favicon: favicon
        };
        if (!panel.bookmarks) {
            panel.bookmarks = [];
        }
        panel.bookmarks.push(newBookmark);
    }

    renderPanels();
    closeBookmarkModal();
    saveData();
}

// Delete bookmark
function deleteBookmark(panelId, bookmarkId) {
    const panel = panelData.find(p => p.id === panelId);
    if (panel) {
        panel.bookmarks = panel.bookmarks.filter(b => b.id !== bookmarkId);
        renderPanels();
        saveData();
    }
}

// Sort all bookmarks by name
function sortAllBookmarks() {
    panelData.forEach(panel => {
        if (panel.bookmarks && panel.bookmarks.length > 0) {
            panel.bookmarks.sort((a, b) => {
                const nameA = (a.title || 'Untitled').toLowerCase();
                const nameB = (b.title || 'Untitled').toLowerCase();
                return nameA.localeCompare(nameB);
            });
        }
    });
    renderPanels();
    saveData();
    alert('All bookmarks sorted by name!');
}

// Fetch bookmark metadata (favicon, title, description)
async function fetchBookmarkMetadata(url) {
    if (!url) return;

    // Ensure URL has protocol
    if (!url.startsWith('http://') && !url.startsWith('https://')) {
        url = 'https://' + url;
        document.getElementById('bookmarkUrl').value = url;
    }

    try {
        // Show loading state
        const titleInput = document.getElementById('bookmarkTitle');
        const descInput = document.getElementById('bookmarkDescription');
        const faviconInput = document.getElementById('bookmarkFavicon');
        
        const originalTitlePlaceholder = titleInput.placeholder;
        const originalDescPlaceholder = descInput.placeholder;
        
        titleInput.placeholder = 'Fetching...';
        descInput.placeholder = 'Fetching...';

        // Fetch favicon
        const favicon = await fetchFavicon(url);
        if (favicon && !faviconInput.value) {
            faviconInput.value = favicon;
        }

        // Fetch title and description
        try {
            const response = await fetch(url, {
                method: 'GET',
                mode: 'cors',
                headers: {
                    'Accept': 'text/html'
                }
            });
            
            if (response.ok) {
                const html = await response.text();
                const parser = new DOMParser();
                const doc = parser.parseFromString(html, 'text/html');
                
                // Extract title
                const title = doc.querySelector('title')?.textContent || 
                             doc.querySelector('meta[property="og:title"]')?.content || '';
                if (title && !titleInput.value) {
                    titleInput.value = title.trim();
                }
                
                // Extract description
                const description = doc.querySelector('meta[name="description"]')?.content ||
                                  doc.querySelector('meta[property="og:description"]')?.content || '';
                if (description && !descInput.value) {
                    descInput.value = description.trim();
                }
            }
                 } catch (error) {
             // CORS or other error - leave fields empty for manual entry
         }

        // Restore placeholders
        titleInput.placeholder = originalTitlePlaceholder;
        descInput.placeholder = originalDescPlaceholder;

         } catch (error) {
         // Error fetching metadata - user can enter manually
     }
}

// Fetch favicon with multiple fallback methods
async function fetchFavicon(url) {
    try {
        const domain = extractDomain(url);
        
        // For IP addresses, localhost, or invalid domains, skip favicon fetching
        // These will show a default icon instead of generating 404s
        if (domain.match(/^\d+\.\d+\.\d+\.\d+$/) || domain.includes('localhost') || !domain.includes('.')) {
            // Return a data URI for a default icon instead of making a request
            return null; // Return null to use emoji fallback in bookmark display
        }
        
        // Use Google's favicon service - it returns a default icon for missing favicons
        // This avoids 404s because Google handles missing favicons gracefully
        const googleFavicon = `https://www.google.com/s2/favicons?domain=${domain}&sz=64`;
        return googleFavicon;
        
    } catch (error) {
        // If domain extraction fails, return null to use fallback
        return null;
    }
}

// Extract domain from URL
function extractDomain(url) {
    try {
        const urlObj = new URL(url.startsWith('http') ? url : `https://${url}`);
        return urlObj.hostname.replace('www.', '');
    } catch (error) {
        // Fallback: simple extraction
        return url.replace(/^https?:\/\//, '').replace(/^www\./, '').split('/')[0].split('?')[0];
    }
}

// Save all data to localStorage
function saveData() {
    try {
        const data = {
            theme: currentTheme,
            columnCount: columnCount,
            backgroundImage: backgroundImageData,
            panels: panelData,
            panelIdCounter: panelIdCounter,
            bookmarkIdCounter: bookmarkIdCounter,
            hideAddBookmarkButtons: hideAddBookmarkButtons,
            frostedEffectLevel: frostedEffectLevel,
            panelDarknessLevel: panelDarknessLevel,
            iconConfigs: iconConfigs,
            version: '1.0'
        };
                 localStorage.setItem('browserHomepageData', JSON.stringify(data));
     } catch (error) {
         alert('Error saving data. Storage may be full.');
     }
}

// Load all data from localStorage
function loadData() {
    try {
        const savedData = localStorage.getItem('browserHomepageData');
        if (savedData) {
            const data = JSON.parse(savedData);
            
            // Load icon configs from saved data (or keep empty arrays on first load)
            if (data.iconConfigs) {
                // Load saved icon configs
                iconConfigs = data.iconConfigs;
                // Ensure all properties exist
                if (!iconConfigs.left) iconConfigs.left = [];
                if (!iconConfigs.right) iconConfigs.right = [];
                if (!iconConfigs.googleApps) iconConfigs.googleApps = [];
            } else {
                // First load - keep empty arrays (no default icons)
                iconConfigs = {
                    left: [],
                    right: [],
                    googleApps: []
                };
            }
            
            if (data.theme) currentTheme = data.theme;
            if (data.columnCount) columnCount = data.columnCount;
            
            // Restore manually uploaded background if available (works with both themes)
            // If no manual background in glass theme, random selection will be used when theme is set
            if (data.backgroundImage) {
                backgroundImageData = data.backgroundImage;
                // Only apply if we're in glass theme (colorful theme doesn't use background images)
                if (currentTheme === 'glass') {
                    const bgImage = document.getElementById('backgroundImage');
                    if (bgImage && backgroundImageData) {
                        bgImage.src = backgroundImageData;
                        bgImage.style.display = 'block';
                        bgImage.onload = function() {
                            bgImage.classList.add('loaded');
                        };
                    }
                }
            }
            if (data.panels) panelData = data.panels;
            if (data.panelIdCounter) panelIdCounter = data.panelIdCounter;
            if (data.bookmarkIdCounter) bookmarkIdCounter = data.bookmarkIdCounter;
            if (data.hideAddBookmarkButtons !== undefined) {
                hideAddBookmarkButtons = data.hideAddBookmarkButtons;
                const checkbox = document.getElementById('hideAddBookmarkBtns');
                if (checkbox) checkbox.checked = hideAddBookmarkButtons;
            }
            if (data.frostedEffectLevel !== undefined) {
                frostedEffectLevel = data.frostedEffectLevel;
                const slider = document.getElementById('frostedEffectSlider');
                const valueDisplay = document.getElementById('frostedEffectValue');
                if (slider) slider.value = frostedEffectLevel;
                if (valueDisplay) valueDisplay.textContent = `${frostedEffectLevel}px`;
            }
            if (data.panelDarknessLevel !== undefined) {
                panelDarknessLevel = data.panelDarknessLevel;
                const slider = document.getElementById('panelDarknessSlider');
                const valueDisplay = document.getElementById('panelDarknessValue');
                if (slider) slider.value = panelDarknessLevel;
                if (valueDisplay) valueDisplay.textContent = `${panelDarknessLevel}%`;
            }
        } else {
            // No saved data - first load, keep empty arrays (blank page)
            iconConfigs = {
                left: [],
                right: [],
                googleApps: []
            };
        }
        
        
     } catch (error) {
         // Error loading data - use empty arrays (blank state)
         iconConfigs = {
             left: [],
             right: [],
             googleApps: []
         };
     }
}

// Export data as JSON file
function exportData() {
    try {
        const data = {
            theme: currentTheme,
            columnCount: columnCount,
            backgroundImage: backgroundImageData,
            panels: panelData,
            panelIdCounter: panelIdCounter,
            bookmarkIdCounter: bookmarkIdCounter,
            hideAddBookmarkButtons: hideAddBookmarkButtons,
            frostedEffectLevel: frostedEffectLevel,
            panelDarknessLevel: panelDarknessLevel,
            iconConfigs: iconConfigs,
            version: '1.0',
            exportDate: new Date().toISOString()
        };
        
        const dataStr = JSON.stringify(data, null, 2);
        const dataBlob = new Blob([dataStr], { type: 'application/json' });
        const url = URL.createObjectURL(dataBlob);
        
        const link = document.createElement('a');
        link.href = url;
        link.download = `browser-homepage-backup-${new Date().toISOString().split('T')[0]}.json`;
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        URL.revokeObjectURL(url);
        
                 alert('Data exported successfully!');
     } catch (error) {
         alert('Error exporting data.');
     }
}

// Import data from JSON file
function importData(file) {
    const reader = new FileReader();
    reader.onload = function(e) {
        try {
            const data = JSON.parse(e.target.result);
            
            // Validate data structure - panels is required, version is optional for backward compatibility
            if (!data.panels) {
                throw new Error('Invalid data format: missing panels data');
            }
            
            // Confirm import
            if (confirm('This will replace all current data. Are you sure?')) {
                if (data.theme) currentTheme = data.theme;
                if (data.columnCount) columnCount = data.columnCount;
                if (data.backgroundImage) backgroundImageData = data.backgroundImage;
                if (data.panels) panelData = data.panels;
                if (data.panelIdCounter) panelIdCounter = data.panelIdCounter;
                if (data.bookmarkIdCounter) bookmarkIdCounter = data.bookmarkIdCounter;
                if (data.hideAddBookmarkButtons !== undefined) {
                    hideAddBookmarkButtons = data.hideAddBookmarkButtons;
                    const checkbox = document.getElementById('hideAddBookmarkBtns');
                    if (checkbox) checkbox.checked = hideAddBookmarkButtons;
                }
                if (data.frostedEffectLevel !== undefined) {
                    frostedEffectLevel = data.frostedEffectLevel;
                    const slider = document.getElementById('frostedEffectSlider');
                    const valueDisplay = document.getElementById('frostedEffectValue');
                    if (slider) slider.value = frostedEffectLevel;
                    if (valueDisplay) valueDisplay.textContent = `${frostedEffectLevel}px`;
                }
                if (data.panelDarknessLevel !== undefined) {
                    panelDarknessLevel = data.panelDarknessLevel;
                    const slider = document.getElementById('panelDarknessSlider');
                    const valueDisplay = document.getElementById('panelDarknessValue');
                    if (slider) slider.value = panelDarknessLevel;
                    if (valueDisplay) valueDisplay.textContent = `${panelDarknessLevel}%`;
                }
                // Icon configs - handle both old and new formats
                if (data.iconConfigs) {
                    iconConfigs = data.iconConfigs;
                    // Ensure all properties exist
                    if (!iconConfigs.left) iconConfigs.left = [];
                    if (!iconConfigs.right) iconConfigs.right = [];
                    if (!iconConfigs.googleApps) iconConfigs.googleApps = [];
                } else {
                    // Old backup without iconConfigs - keep current empty arrays (blank state)
                    iconConfigs = {
                        left: [],
                        right: [],
                        googleApps: []
                    };
                }
                
                // Update UI
                document.getElementById('themeSelector').value = currentTheme;
                document.getElementById('columnSelector').value = columnCount;
                setTheme(currentTheme);
                updateColumnLayout(columnCount);
                
                if (backgroundImageData) {
                    setBackgroundImage(backgroundImageData);
                }
                
                // Re-render icons with new configs - preserve settings button
                const leftGroup = document.querySelector('.icon-group-left');
                const rightGroup = document.querySelector('.icon-group-right');
                const settingsBtn = document.getElementById('settingsToggleBtn');
                
                if (leftGroup) {
                    const leftIcons = leftGroup.querySelectorAll('.icon-link:not(.settings-toggle-btn), .dropdown-container');
                    leftIcons.forEach(el => el.remove());
                }
                
                if (rightGroup) {
                    const rightIcons = rightGroup.querySelectorAll('.icon-link:not(.settings-toggle-btn)');
                    rightIcons.forEach(el => el.remove());
                }
                
                // Re-add icons
                iconConfigs.left.forEach(icon => {
                    const iconElement = createIconElement(icon);
                    leftGroup.appendChild(iconElement);
                });
                
                const dropdown = createDropdown('▼', iconConfigs.googleApps);
                leftGroup.appendChild(dropdown);
                
                iconConfigs.right.forEach(icon => {
                    const iconElement = createIconElement(icon);
                    rightGroup.appendChild(iconElement);
                });
                
                // Ensure settings button is at the end
                if (settingsBtn && rightGroup && !rightGroup.contains(settingsBtn)) {
                    rightGroup.appendChild(settingsBtn);
                } else if (settingsBtn && rightGroup) {
                    rightGroup.removeChild(settingsBtn);
                    rightGroup.appendChild(settingsBtn);
                }
                
                renderPanels();
                saveData();
                                 alert('Data imported successfully!');
             }
         } catch (error) {
             alert('Error importing data. Please check the file format.');
         }
    };
    reader.readAsText(file);
}



// Initialize settings
function initializeSettings() {
    const settingsToggleBtn = document.getElementById('settingsToggleBtn');
    const settingsBar = document.getElementById('settingsBar');
    const closeSettingsBtn = document.getElementById('closeSettingsBtn');
    const hideAddBookmarkCheckbox = document.getElementById('hideAddBookmarkBtns');
    const addPanelBtn = document.getElementById('addPanelBtn');
    const columnSelector = document.getElementById('columnSelector');
    const themeSelector = document.getElementById('themeSelector');
    const changeBgBtn = document.getElementById('changeBgImageBtn');
    const bgImageInput = document.getElementById('bgImageInput');
    const exportBtn = document.getElementById('exportBtn');
    const importBtn = document.getElementById('importBtn');
    const importInput = document.getElementById('importInput');
    const frostedEffectSlider = document.getElementById('frostedEffectSlider');
    const frostedEffectValue = document.getElementById('frostedEffectValue');
    const frostedEffectSetting = document.getElementById('frostedEffectSetting');
    const panelDarknessSlider = document.getElementById('panelDarknessSlider');
    const panelDarknessValue = document.getElementById('panelDarknessValue');
    const panelDarknessSetting = document.getElementById('panelDarknessSetting');



    // Set initial values
    columnSelector.value = columnCount;
    themeSelector.value = currentTheme;
    frostedEffectSlider.value = frostedEffectLevel;
    frostedEffectValue.textContent = `${frostedEffectLevel}px`;
    panelDarknessSlider.value = panelDarknessLevel;
    panelDarknessValue.textContent = `${panelDarknessLevel}%`;
    
    // Show/hide glass theme specific settings based on theme
    if (frostedEffectSetting) {
        frostedEffectSetting.style.display = currentTheme === 'glass' ? 'block' : 'none';
    }
    if (panelDarknessSetting) {
        panelDarknessSetting.style.display = currentTheme === 'glass' ? 'block' : 'none';
    }
    
    // Apply initial effects
    applyFrostedEffect();
    applyPanelDarkness();

    // Toggle settings bar
    settingsToggleBtn.addEventListener('click', function() {
        settingsBar.classList.toggle('show');
    });

    closeSettingsBtn.addEventListener('click', function() {
        settingsBar.classList.remove('show');
    });

    // Close settings when clicking outside (but not on buttons or interactive elements)
    settingsBar.addEventListener('click', function(e) {
        // Get the clicked element
        const target = e.target;
        
        // Don't close if clicking on:
        // - Buttons
        // - Inputs
        // - Labels
        // - Settings items
        // - Any interactive element
        if (target.tagName === 'BUTTON' || 
            target.tagName === 'INPUT' || 
            target.tagName === 'LABEL' ||
            target.closest('button') ||
            target.closest('input') ||
            target.closest('label') ||
            target.closest('.settings-item') ||
            target.closest('.settings-section')) {
            return; // Don't close settings
        }
        
        // Only close if clicking directly on the settings bar background
        if (e.target === settingsBar) {
            settingsBar.classList.remove('show');
        }
    });

    // Add Panel button
    addPanelBtn.addEventListener('click', function() {
        addPanel();
        saveData();
    });

    // Column selector
    columnSelector.addEventListener('change', function() {
        columnCount = parseInt(this.value);
        updateColumnLayout(columnCount);
        saveData();
    });

    // Theme selector
    themeSelector.addEventListener('change', function() {
        currentTheme = this.value;
        setTheme(currentTheme);
    });

    // Frosted effect slider
    frostedEffectSlider.addEventListener('input', function() {
        frostedEffectLevel = parseInt(this.value);
        frostedEffectValue.textContent = `${frostedEffectLevel}px`;
        applyFrostedEffect();
        saveData();
    });

    // Panel darkness slider
    panelDarknessSlider.addEventListener('input', function() {
        panelDarknessLevel = parseInt(this.value);
        panelDarknessValue.textContent = `${panelDarknessLevel}%`;
        applyPanelDarkness();
        saveData();
    });

    // Change background image
    changeBgBtn.addEventListener('click', function() {
        bgImageInput.click();
    });

    bgImageInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file && file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = function(event) {
                const imageDataUrl = event.target.result;
                setBackgroundImage(imageDataUrl);
            };
            reader.readAsDataURL(file);
        }
    });

    // Export button
    exportBtn.addEventListener('click', exportData);

    // Import button
    importBtn.addEventListener('click', function() {
        importInput.click();
    });

    importInput.addEventListener('change', function(e) {
        if (e.target.files[0]) {
            importData(e.target.files[0]);
        }
    });

    // Reset button - use global resetAllData function
    const resetBtn = document.getElementById('resetBtn');
    if (resetBtn && typeof window.resetAllData === 'function') {
        // Replace button to remove any existing handlers
        const newBtn = resetBtn.cloneNode(true);
        resetBtn.parentNode.replaceChild(newBtn, resetBtn);
        
        // Attach global function directly
        newBtn.onclick = window.resetAllData;
        newBtn.addEventListener('click', window.resetAllData, false);
        
    } else if (resetBtn) {
        // Fallback if global function not available
        resetBtn.onclick = function() {
            if (confirm('⚠️ WARNING: Reset all data to defaults?')) {
                localStorage.removeItem('browserHomepageData');
                localStorage.removeItem('browserHomepageSetupCompleted');
                location.reload();
            }
        };
    }

    // Handle hide add bookmark buttons setting
    hideAddBookmarkCheckbox.addEventListener('change', function() {
        hideAddBookmarkButtons = this.checked;
        renderPanels(); // Re-render panels to apply the change
        saveData();
    });

    // Sort bookmarks button
    const sortBookmarksBtn = document.getElementById('sortBookmarksBtn');
    if (sortBookmarksBtn) {
        sortBookmarksBtn.addEventListener('click', sortAllBookmarks);
    }

    // Setup wizard button
    const setupWizardBtn = document.getElementById('setupWizardBtn');
    if (setupWizardBtn) {
        setupWizardBtn.addEventListener('click', function() {
            openSetupWizard();
        });
    }

    // Tutorial wizard button
    const showTutorialBtn = document.getElementById('showTutorialBtn');
    if (showTutorialBtn) {
        showTutorialBtn.addEventListener('click', function() {
            openTutorialWizard();
        });
    }

    // About button
    const aboutBtn = document.getElementById('aboutBtn');
    if (aboutBtn) {
        aboutBtn.addEventListener('click', function() {
            openAboutModal();
        });
    }
    
    // Update copyright year when modal opens
    const copyrightYear = document.getElementById('copyrightYear');
    if (copyrightYear) {
        copyrightYear.textContent = new Date().getFullYear();
    }
}

// Setup Wizard Functions
let currentSetupTab = 'left';
let setupWizardData = null;

function openSetupWizard() {
    const modal = document.getElementById('setupWizardModal');
    if (!modal) return;
    
    // Clone current icon configs for editing
    setupWizardData = {
        left: JSON.parse(JSON.stringify(iconConfigs.left)),
        right: JSON.parse(JSON.stringify(iconConfigs.right)),
        googleApps: JSON.parse(JSON.stringify(iconConfigs.googleApps))
    };
    
    currentSetupTab = 'left';
    renderSetupTab('left');
    modal.classList.add('show');
    
    // Attach save button listener when modal opens (in case it was recreated)
    const saveBtn = document.getElementById('saveSetupWizard');
    if (saveBtn) {
        // Remove old listener by cloning
        const newSaveBtn = saveBtn.cloneNode(true);
        saveBtn.parentNode.replaceChild(newSaveBtn, saveBtn);
        
        newSaveBtn.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            saveSetupWizard();
            return false;
        });
    }
}

function closeSetupWizard() {
    const modal = document.getElementById('setupWizardModal');
    if (modal) {
        modal.classList.remove('show');
    }
    setupWizardData = null;
}

function renderSetupTab(tabName) {
    currentSetupTab = tabName;
    const tabContent = document.getElementById('setupTabContent');
    if (!tabContent || !setupWizardData) return;
    
    // Update active tab
    document.querySelectorAll('.setup-tab').forEach(tab => {
        tab.classList.remove('active');
        if (tab.dataset.tab === tabName) {
            tab.classList.add('active');
        }
    });
    
    // Clear content
    tabContent.innerHTML = '';
    
    // Get icons for current tab
    const icons = setupWizardData[tabName] || [];
    
    // Render each icon
    icons.forEach((icon, index) => {
        const iconItem = createSetupIconItem(icon, index, tabName);
        tabContent.appendChild(iconItem);
    });
    
    // Add "Add Icon" button
    const addBtn = document.createElement('button');
    addBtn.className = 'setup-add-icon';
    addBtn.textContent = '+ Add Icon';
    addBtn.addEventListener('click', function() {
        addSetupIcon(tabName);
    });
    tabContent.appendChild(addBtn);
}

function createSetupIconItem(icon, index, tabName) {
    const item = document.createElement('div');
    item.className = 'setup-icon-item';
    
    // Icon preview
    const preview = document.createElement('div');
    preview.className = 'icon-preview';
    
    // Function to update preview
    function updatePreview() {
        preview.innerHTML = '';
        // Priority: customIcon > iconFilename > getIconFilename(name) > emoji
        if (setupWizardData[tabName][index].customIcon) {
            // Use custom uploaded icon
            const img = document.createElement('img');
            img.src = setupWizardData[tabName][index].customIcon;
            img.alt = setupWizardData[tabName][index].name || '';
            img.onerror = function() {
                const span = document.createElement('span');
                span.className = 'icon-emoji';
                span.textContent = setupWizardData[tabName][index].icon || '🔗';
                preview.innerHTML = '';
                preview.appendChild(span);
            };
            preview.appendChild(img);
        } else if (setupWizardData[tabName][index].iconFilename) {
            // Use selected icon - check if it's a URL (CoreUI) or local file
            const iconSrc = setupWizardData[tabName][index].iconFilename;
            const isUrl = iconSrc.startsWith('http://') || iconSrc.startsWith('https://');
            const img = document.createElement('img');
            img.src = isUrl ? iconSrc : `Icons/${iconSrc}`;
            img.alt = setupWizardData[tabName][index].name || '';
            img.onerror = function() {
                const span = document.createElement('span');
                span.className = 'icon-emoji';
                span.textContent = setupWizardData[tabName][index].icon || '🔗';
                preview.innerHTML = '';
                preview.appendChild(span);
            };
            preview.appendChild(img);
        } else {
            // Check for auto-detected filename or emoji
            const iconFilename = getIconFilename(setupWizardData[tabName][index].name);
            if (iconFilename) {
                const img = document.createElement('img');
                img.src = `Icons/${iconFilename}`;
                img.alt = setupWizardData[tabName][index].name || '';
                img.onerror = function() {
                    const span = document.createElement('span');
                    span.className = 'icon-emoji';
                    span.textContent = setupWizardData[tabName][index].icon || '🔗';
                    preview.innerHTML = '';
                    preview.appendChild(span);
                };
                preview.appendChild(img);
            } else {
                const span = document.createElement('span');
                span.className = 'icon-emoji';
                span.textContent = setupWizardData[tabName][index].icon || '🔗';
                preview.appendChild(span);
            }
        }
    }
    
    updatePreview();
    
    // Fields
    const fields = document.createElement('div');
    fields.className = 'icon-fields';
    
    const nameInput = document.createElement('input');
    nameInput.type = 'text';
    nameInput.placeholder = 'Icon Name';
    nameInput.value = icon.name || '';
    nameInput.addEventListener('input', function() {
        setupWizardData[tabName][index].name = this.value;
        updatePreview();
    });
    
    const urlInput = document.createElement('input');
    urlInput.type = 'url';
    urlInput.placeholder = 'URL (e.g., https://example.com)';
    urlInput.value = icon.url || '';
    urlInput.addEventListener('input', function() {
        setupWizardData[tabName][index].url = this.value;
    });
    
    // Icon selection dropdown
    const iconSelectContainer = document.createElement('div');
    iconSelectContainer.className = 'icon-select-container';
    
    const iconSelectLabel = document.createElement('label');
    iconSelectLabel.className = 'icon-select-label';
    iconSelectLabel.textContent = 'Choose Icon:';
    iconSelectLabel.style.fontSize = '12px';
    iconSelectLabel.style.color = 'rgba(255, 255, 255, 0.7)';
    iconSelectLabel.style.marginBottom = '4px';
    iconSelectLabel.style.display = 'block';
    
    const iconSelect = document.createElement('select');
    iconSelect.className = 'icon-select';
    iconSelect.style.width = '100%';
    iconSelect.style.padding = '6px 10px';
    iconSelect.style.background = 'rgba(255, 255, 255, 0.1)';
    iconSelect.style.border = '1px solid rgba(255, 255, 255, 0.2)';
    iconSelect.style.borderRadius = '6px';
    iconSelect.style.color = '#fff';
    iconSelect.style.fontSize = '14px';
    iconSelect.style.cursor = 'pointer';
    
    // Add default option
    const defaultOption = document.createElement('option');
    defaultOption.value = '';
    defaultOption.textContent = 'Auto (based on name)';
    iconSelect.appendChild(defaultOption);
    
    // Add "None (use emoji)" option
    const noneOption = document.createElement('option');
    noneOption.value = '__none__';
    noneOption.textContent = 'None (use emoji)';
    iconSelect.appendChild(noneOption);
    
    // Add separator for local icons
    const localSeparator = document.createElement('option');
    localSeparator.disabled = true;
    localSeparator.textContent = '─── Local Icons ───';
    iconSelect.appendChild(localSeparator);
    
    // Add all available local icons
    availableIcons.forEach(filename => {
        const option = document.createElement('option');
        option.value = filename;
        option.textContent = filename.replace('.png', '').replace(/\s+/g, ' ');
        if (icon.iconFilename === filename) {
            option.selected = true;
        }
        iconSelect.appendChild(option);
    });
    
    // Add separator for CoreUI icons
    const coreUISeparator = document.createElement('option');
    coreUISeparator.disabled = true;
    coreUISeparator.textContent = '─── CoreUI Icons ───';
    iconSelect.appendChild(coreUISeparator);
    
    // Add CoreUI icons
    coreUIIcons.forEach(coreIcon => {
        const option = document.createElement('option');
        option.value = coreIcon.url; // Store the full URL
        option.textContent = `CoreUI: ${coreIcon.name}`;
        if (icon.iconFilename === coreIcon.url || icon.iconUrl === coreIcon.url) {
            option.selected = true;
        }
        iconSelect.appendChild(option);
    });
    
    // Set current selection
    if (icon.iconFilename) {
        // Check if it's a URL (CoreUI icon) or local file
        if (icon.iconFilename.startsWith('http://') || icon.iconFilename.startsWith('https://')) {
            iconSelect.value = icon.iconFilename;
        } else {
            iconSelect.value = icon.iconFilename;
        }
    } else if (icon.iconUrl) {
        // Legacy support for iconUrl property
        iconSelect.value = icon.iconUrl;
    } else if (icon.customIcon) {
        iconSelect.value = '__custom__';
        const customOption = document.createElement('option');
        customOption.value = '__custom__';
        customOption.textContent = 'Custom Uploaded Icon';
        customOption.selected = true;
        iconSelect.insertBefore(customOption, iconSelect.firstChild);
    }
    
    iconSelect.addEventListener('change', function() {
        if (this.value === '__none__') {
            setupWizardData[tabName][index].iconFilename = null;
            delete setupWizardData[tabName][index].iconFilename;
            setupWizardData[tabName][index].iconUrl = null;
            delete setupWizardData[tabName][index].iconUrl;
            // Clear custom icon too if switching to none
            setupWizardData[tabName][index].customIcon = null;
            delete setupWizardData[tabName][index].customIcon;
            iconUploadLabel.textContent = 'Upload Custom Icon';
            if (iconRemoveBtn) iconRemoveBtn.style.display = 'none';
        } else if (this.value === '') {
            // Auto mode - clear iconFilename and iconUrl
            setupWizardData[tabName][index].iconFilename = null;
            delete setupWizardData[tabName][index].iconFilename;
            setupWizardData[tabName][index].iconUrl = null;
            delete setupWizardData[tabName][index].iconUrl;
        } else if (this.value.startsWith('http://') || this.value.startsWith('https://')) {
            // CoreUI icon (remote URL) - store as iconFilename for compatibility
            setupWizardData[tabName][index].iconFilename = this.value;
            setupWizardData[tabName][index].iconUrl = this.value;
            // Clear custom icon when selecting remote icon
            setupWizardData[tabName][index].customIcon = null;
            delete setupWizardData[tabName][index].customIcon;
            iconUploadLabel.textContent = 'Upload Custom Icon';
            if (iconRemoveBtn) iconRemoveBtn.style.display = 'none';
        } else {
            // Local icon file
            setupWizardData[tabName][index].iconFilename = this.value;
            setupWizardData[tabName][index].iconUrl = null;
            delete setupWizardData[tabName][index].iconUrl;
            // Clear custom icon when selecting from folder
            setupWizardData[tabName][index].customIcon = null;
            delete setupWizardData[tabName][index].customIcon;
            iconUploadLabel.textContent = 'Upload Custom Icon';
            if (iconRemoveBtn) iconRemoveBtn.style.display = 'none';
        }
        updatePreview();
    });
    
    iconSelectContainer.appendChild(iconSelectLabel);
    iconSelectContainer.appendChild(iconSelect);
    
    // Custom icon upload
    const iconUploadContainer = document.createElement('div');
    iconUploadContainer.className = 'icon-upload-container';
    
    const iconUploadLabel = document.createElement('label');
    iconUploadLabel.className = 'icon-upload-label';
    iconUploadLabel.textContent = icon.customIcon ? 'Change Custom Icon' : 'Upload Custom Icon';
    iconUploadLabel.style.cursor = 'pointer';
    
    const iconUploadInput = document.createElement('input');
    iconUploadInput.type = 'file';
    iconUploadInput.accept = 'image/*';
    iconUploadInput.style.display = 'none';
    iconUploadInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file && file.type.startsWith('image/')) {
            const reader = new FileReader();
            reader.onload = function(event) {
                setupWizardData[tabName][index].customIcon = event.target.result;
                // Clear iconFilename when uploading custom icon
                setupWizardData[tabName][index].iconFilename = null;
                delete setupWizardData[tabName][index].iconFilename;
                // Update dropdown to show custom
                if (iconSelect.value !== '__custom__') {
                    const customOption = document.createElement('option');
                    customOption.value = '__custom__';
                    customOption.textContent = 'Custom Uploaded Icon';
                    customOption.selected = true;
                    iconSelect.insertBefore(customOption, iconSelect.firstChild);
                    iconSelect.value = '__custom__';
                }
                updatePreview();
                iconUploadLabel.textContent = 'Change Custom Icon';
                if (iconRemoveBtn) {
                    iconRemoveBtn.style.display = 'inline-block';
                }
            };
            reader.readAsDataURL(file);
        }
    });
    
    iconUploadLabel.addEventListener('click', function() {
        iconUploadInput.click();
    });
    
    iconUploadContainer.appendChild(iconUploadLabel);
    iconUploadContainer.appendChild(iconUploadInput);
    
    // Remove custom icon button
    const iconRemoveBtn = document.createElement('button');
    iconRemoveBtn.type = 'button';
    iconRemoveBtn.className = 'icon-remove-custom';
    iconRemoveBtn.textContent = 'Remove Custom Icon';
    iconRemoveBtn.style.display = icon.customIcon ? 'inline-block' : 'none';
    iconRemoveBtn.addEventListener('click', function() {
        setupWizardData[tabName][index].customIcon = null;
        delete setupWizardData[tabName][index].customIcon;
        // Reset dropdown to auto
        iconSelect.value = '';
        updatePreview();
        iconUploadLabel.textContent = 'Upload Custom Icon';
        iconRemoveBtn.style.display = 'none';
        iconUploadInput.value = '';
    });
    
    iconUploadContainer.appendChild(iconRemoveBtn);
    
    fields.appendChild(nameInput);
    fields.appendChild(urlInput);
    fields.appendChild(iconSelectContainer);
    fields.appendChild(iconUploadContainer);
    
    // Remove button
    const removeBtn = document.createElement('button');
    removeBtn.className = 'icon-remove';
    removeBtn.textContent = 'Remove';
    removeBtn.addEventListener('click', function() {
        setupWizardData[tabName].splice(index, 1);
        renderSetupTab(tabName);
    });
    
    item.appendChild(preview);
    item.appendChild(fields);
    item.appendChild(removeBtn);
    
    return item;
}

function addSetupIcon(tabName) {
    if (!setupWizardData) return;
    
    const newIcon = {
        name: '',
        url: '',
        icon: '🔗'
    };
    
    setupWizardData[tabName].push(newIcon);
    renderSetupTab(tabName);
}

function saveSetupWizard() {
    if (!setupWizardData) {
        alert('No data to save. Please try opening the setup wizard again.');
        return false;
    }
    
    // Filter out icons that don't have both name and URL, but preserve all properties
    const filterValidIcons = (icons) => {
        return icons.filter(icon => {
            const hasName = icon.name && icon.name.trim().length > 0;
            const hasUrl = icon.url && icon.url.trim().length > 0;
            return hasName && hasUrl;
        }).map(icon => {
            // Preserve all icon properties including customIcon, iconFilename, and icon
            const preservedIcon = {
                name: icon.name,
                url: icon.url,
                icon: icon.icon || '🔗'
            };
            // Preserve customIcon if it exists (even if it's an empty string, preserve it)
            if (icon.hasOwnProperty('customIcon') && icon.customIcon) {
                preservedIcon.customIcon = icon.customIcon;
            }
            // Preserve iconFilename if it exists (even if it's an empty string, preserve it)
            if (icon.hasOwnProperty('iconFilename') && icon.iconFilename) {
                preservedIcon.iconFilename = icon.iconFilename;
            }
            // Preserve iconUrl if it exists (for CoreUI icons)
            if (icon.hasOwnProperty('iconUrl') && icon.iconUrl) {
                preservedIcon.iconUrl = icon.iconUrl;
            }
            return preservedIcon;
        });
    };
    
    // Update icon configs with only valid icons (preserving all properties)
    iconConfigs.left = filterValidIcons(setupWizardData.left);
    iconConfigs.right = filterValidIcons(setupWizardData.right);
    iconConfigs.googleApps = filterValidIcons(setupWizardData.googleApps);
    
    // Allow saving even with no icons (user can have an empty top bar)
    
    // Re-render icons - preserve settings button
    const leftGroup = document.querySelector('.icon-group-left');
    const rightGroup = document.querySelector('.icon-group-right');
    const settingsBtn = document.getElementById('settingsToggleBtn');
    
    if (leftGroup) {
        // Clear only icon elements, not the entire group
        const leftIcons = leftGroup.querySelectorAll('.icon-link:not(.settings-toggle-btn), .dropdown-container');
        leftIcons.forEach(el => el.remove());
    }
    
    if (rightGroup) {
        // Clear only icon elements, preserve settings button
        const rightIcons = rightGroup.querySelectorAll('.icon-link:not(.settings-toggle-btn)');
        rightIcons.forEach(el => el.remove());
    }
    
    // Re-add icons
    iconConfigs.left.forEach(icon => {
        const iconElement = createIconElement(icon);
        leftGroup.appendChild(iconElement);
    });
    
    // Re-add dropdown
    const dropdown = createDropdown('▼', iconConfigs.googleApps);
    leftGroup.appendChild(dropdown);
    
    // Re-add right icons
    iconConfigs.right.forEach(icon => {
        const iconElement = createIconElement(icon);
        rightGroup.appendChild(iconElement);
    });
    
    // Ensure settings button is at the end
    if (settingsBtn && rightGroup && !rightGroup.contains(settingsBtn)) {
        rightGroup.appendChild(settingsBtn);
    } else if (settingsBtn && rightGroup) {
        // Move to end if already present
        rightGroup.removeChild(settingsBtn);
        rightGroup.appendChild(settingsBtn);
    }
    
    // Save to localStorage
    saveData();
    
    // Mark setup as completed
    localStorage.setItem('browserHomepageSetupCompleted', 'true');
    
    // Close wizard
    closeSetupWizard();
    
    alert('Icon links updated successfully!');
    return true;
}

// Make function globally accessible for onclick handler
window.saveSetupWizard = saveSetupWizard;

// Initialize setup wizard
function initializeSetupWizard() {
    const modal = document.getElementById('setupWizardModal');
    const closeBtn = document.getElementById('closeSetupWizard');
    const cancelBtn = document.getElementById('cancelSetupWizard');
    const saveBtn = document.getElementById('saveSetupWizard');
    const tabs = document.querySelectorAll('.setup-tab');
    
    if (!modal) return;
    
    // Tab switching
    tabs.forEach(tab => {
        tab.addEventListener('click', function() {
            const tabName = this.dataset.tab;
            if (setupWizardData) {
                renderSetupTab(tabName);
            }
        });
    });
    
    // Close buttons
    if (closeBtn) {
        closeBtn.addEventListener('click', closeSetupWizard);
    }
    if (cancelBtn) {
        cancelBtn.addEventListener('click', closeSetupWizard);
    }
    
    // Save button - use event delegation to ensure it works
    if (saveBtn) {
        // Remove any existing listeners by cloning
        const newSaveBtn = saveBtn.cloneNode(true);
        saveBtn.parentNode.replaceChild(newSaveBtn, saveBtn);
        
        newSaveBtn.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            saveSetupWizard();
            return false;
        });
    }
    
    // Also attach to modal content to catch clicks
    const modalContent = modal.querySelector('.modal-content');
    if (modalContent) {
        modalContent.addEventListener('click', function(e) {
            if (e.target && e.target.id === 'saveSetupWizard') {
                e.preventDefault();
                e.stopPropagation();
                saveSetupWizard();
                return false;
            }
        });
    }
    
    // Close on outside click
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeSetupWizard();
        }
    });
    
    // Check for first-time load - show tutorial wizard instead of just icon setup
    const hasSetupCompleted = localStorage.getItem('browserHomepageSetupCompleted');
    if (!hasSetupCompleted) {
        // Show tutorial wizard on first load
        setTimeout(function() {
            openTutorialWizard();
        }, 500);
    }
}

// About Modal Functions
function openAboutModal() {
    const modal = document.getElementById('aboutModal');
    if (!modal) return;
    
    // Update copyright year
    const copyrightYear = document.getElementById('copyrightYear');
    if (copyrightYear) {
        copyrightYear.textContent = new Date().getFullYear();
    }
    
    modal.classList.add('show');
}

function closeAboutModal() {
    const modal = document.getElementById('aboutModal');
    if (!modal) return;
    modal.classList.remove('show');
}

// Tutorial Wizard Functions
let currentTutorialStep = 0;
const tutorialSteps = [
    {
        title: 'Welcome to SFLDashboard!',
        content: `
            <h3>Welcome! 👋</h3>
            <p>This tutorial will guide you through setting up your personalized SFLDashboard. Don't worry - it's easy!</p>
            <p>You'll learn how to:</p>
            <ul>
                <li>Add quick links to your favorite websites at the top</li>
                <li>Create panels to organize your bookmarks</li>
                <li>Add bookmarks to your panels</li>
                <li>Customize themes and settings</li>
            </ul>
            <p>Let's get started! Click <span class="tutorial-highlight">Next</span> to continue.</p>
        `
    },
    {
        title: 'Step 1: Add Top Navigation Icons',
        content: `
            <h3>Quick Access Icons</h3>
            <p>The top bar lets you add quick links to websites you visit often. You can add icons on the <span class="tutorial-highlight">left</span> and <span class="tutorial-highlight">right</span> sides of the search bar.</p>
            <p><strong>How to add icons:</strong></p>
            <ol>
                <li>Click the <span class="tutorial-button-demo">⚙️ Customize Icon Links</span> button in Settings (gear icon)</li>
                <li>Choose which side (Left, Right, or Other Apps dropdown)</li>
                <li>Click <span class="tutorial-button-demo">+ Add Icon</span></li>
                <li>Enter the website name and URL</li>
                <li>Choose an icon from the library or upload your own</li>
                <li>Click <span class="tutorial-button-demo">Save Changes</span></li>
            </ol>
            <p><strong>Tip:</strong> You can skip this step and add icons later if you want!</p>
        `
    },
    {
        title: 'Step 2: Create Your First Panel',
        content: `
            <h3>Organize with Panels</h3>
            <p>Panels are like folders that help you organize your bookmarks into categories (like "Work", "Entertainment", "Shopping", etc.).</p>
            <p><strong>How to create a panel:</strong></p>
            <ol>
                <li>Click the gear icon <span class="tutorial-highlight">⚙️</span> in the top right corner</li>
                <li>In the Settings panel, find the <span class="tutorial-button-demo">+ Add Panel</span> button</li>
                <li>Click it - a new panel will appear!</li>
                <li>Click on the panel title to rename it (e.g., "Work", "Shopping", "Entertainment")</li>
            </ol>
            <p><strong>Tip:</strong> You can have multiple panels to organize different types of bookmarks!</p>
        `
    },
    {
        title: 'Step 3: Add Your First Bookmark',
        content: `
            <h3>Adding Bookmarks</h3>
            <p>Now let's add a bookmark to your panel! Bookmarks are links to websites you want to save.</p>
            <p><strong>How to add a bookmark:</strong></p>
            <ol>
                <li>Find the panel where you want to add a bookmark</li>
                <li>Click the <span class="tutorial-button-demo">+ Add Bookmark</span> button on that panel</li>
                <li>Enter the website URL (like <span class="tutorial-highlight">https://example.com</span>)</li>
                <li>The title and description will be filled in automatically</li>
                <li>Select which panel to add it to from the dropdown</li>
                <li>Click <span class="tutorial-button-demo">Add Bookmark</span></li>
            </ol>
            <p><strong>Tip:</strong> You can drag and drop bookmarks between panels to reorganize them!</p>
        `
    },
    {
        title: 'Step 4: Choose Your Theme',
        content: `
            <h3>Customize the Look</h3>
            <p>Your homepage has two beautiful themes to choose from:</p>
            <ul>
                <li><strong>Glass Theme:</strong> Frosted glass panels with beautiful background images that change randomly</li>
                <li><strong>Colorful Theme:</strong> Vibrant panels with colorful borders and an animated gradient background</li>
            </ul>
            <p><strong>How to change themes:</strong></p>
            <ol>
                <li>Click the gear icon <span class="tutorial-highlight">⚙️</span></li>
                <li>Find the <span class="tutorial-highlight">Theme</span> dropdown</li>
                <li>Select "Glass" or "Colorful"</li>
                <li>The theme changes instantly!</li>
            </ol>
            <p><strong>Glass Theme Settings:</strong> You can adjust the frosted effect and panel darkness for better visibility.</p>
        `
    },
    {
        title: 'Step 5: Explore Settings',
        content: `
            <h3>More Customization Options</h3>
            <p>The Settings panel (gear icon) has many options to customize your homepage:</p>
            <ul>
                <li><strong>Columns:</strong> Choose 3-6 columns for your panel layout</li>
                <li><strong>Hide "Add Bookmark" buttons:</strong> Clean up the look by hiding these buttons</li>
                <li><strong>Sort Bookmarks:</strong> Alphabetically organize all your bookmarks</li>
                <li><strong>Change Background:</strong> Upload your own background image (Glass theme)</li>
                <li><strong>Export/Import:</strong> Backup your data or restore from a backup</li>
                <li><strong>Reset:</strong> Start over with a clean slate</li>
            </ul>
            <p><strong>Tip:</strong> All your settings are saved automatically!</p>
        `
    },
    {
        title: 'You\'re All Set! 🎉',
        content: `
            <h3>Congratulations!</h3>
            <p>You now know how to use your new homepage! Here's a quick recap:</p>
            <ul>
                <li>✅ Add icons to the top bar for quick access</li>
                <li>✅ Create panels to organize bookmarks</li>
                <li>✅ Add bookmarks to your panels</li>
                <li>✅ Customize themes and settings</li>
            </ul>
            <p><strong>Need help later?</strong></p>
            <p>You can always access the icon setup wizard from Settings → <span class="tutorial-button-demo">⚙️ Customize Icon Links</span></p>
            <p>Enjoy your new personalized homepage! 🚀</p>
        `
    }
];

function openTutorialWizard() {
    const modal = document.getElementById('tutorialWizardModal');
    if (!modal) return;
    
    currentTutorialStep = 0;
    renderTutorialStep(0);
    modal.classList.add('show');
}

function closeTutorialWizard(skipMarkingComplete = false) {
    const modal = document.getElementById('tutorialWizardModal');
    if (modal) {
        modal.classList.remove('show');
    }
    // Mark tutorial as completed only if we finished it (not if skipped)
    if (!skipMarkingComplete && currentTutorialStep === tutorialSteps.length - 1) {
        localStorage.setItem('browserHomepageSetupCompleted', 'true');
    }
}

function renderTutorialStep(stepIndex) {
    if (stepIndex < 0 || stepIndex >= tutorialSteps.length) return;
    
    currentTutorialStep = stepIndex;
    const step = tutorialSteps[stepIndex];
    
    // Update title
    const titleEl = document.getElementById('tutorialTitle');
    if (titleEl) titleEl.textContent = step.title;
    
    // Update content
    const contentEl = document.getElementById('tutorialContent');
    if (contentEl) {
        contentEl.innerHTML = `<div class="tutorial-step active">${step.content}</div>`;
    }
    
    // Update progress indicator
    const progressEl = document.getElementById('tutorialProgress');
    if (progressEl) {
        progressEl.innerHTML = '';
        for (let i = 0; i < tutorialSteps.length; i++) {
            const dot = document.createElement('div');
            dot.className = 'tutorial-step-dot';
            if (i === stepIndex) {
                dot.classList.add('active');
            } else if (i < stepIndex) {
                dot.classList.add('completed');
            }
            progressEl.appendChild(dot);
        }
    }
    
    // Update buttons
    const prevBtn = document.getElementById('prevTutorialStep');
    const nextBtn = document.getElementById('nextTutorialStep');
    const finishBtn = document.getElementById('finishTutorial');
    const skipBtn = document.getElementById('skipTutorial');
    
    if (prevBtn) {
        prevBtn.style.display = stepIndex > 0 ? 'inline-block' : 'none';
    }
    if (nextBtn) {
        nextBtn.style.display = stepIndex < tutorialSteps.length - 1 ? 'inline-block' : 'none';
    }
    if (finishBtn) {
        finishBtn.style.display = stepIndex === tutorialSteps.length - 1 ? 'inline-block' : 'none';
    }
    if (skipBtn) {
        skipBtn.style.display = stepIndex === 0 ? 'inline-block' : 'none';
    }
}

function nextTutorialStep() {
    if (currentTutorialStep < tutorialSteps.length - 1) {
        renderTutorialStep(currentTutorialStep + 1);
    }
}

function prevTutorialStep() {
    if (currentTutorialStep > 0) {
        renderTutorialStep(currentTutorialStep - 1);
    }
}

function finishTutorial() {
    // Mark tutorial as completed
    localStorage.setItem('browserHomepageSetupCompleted', 'true');
    closeTutorialWizard(false);
    // Optionally open icon setup wizard
    setTimeout(function() {
        if (confirm('Would you like to set up your top navigation icons now?')) {
            openSetupWizard();
        }
    }, 300);
}

function initializeTutorialWizard() {
    const modal = document.getElementById('tutorialWizardModal');
    if (!modal) return;
    
    const closeBtn = document.getElementById('closeTutorialWizard');
    const skipBtn = document.getElementById('skipTutorial');
    const prevBtn = document.getElementById('prevTutorialStep');
    const nextBtn = document.getElementById('nextTutorialStep');
    const finishBtn = document.getElementById('finishTutorial');
    
    if (closeBtn) {
        closeBtn.addEventListener('click', function() {
            closeTutorialWizard(true); // Don't mark as completed if closed via X button
        });
    }
    
    if (skipBtn) {
        skipBtn.addEventListener('click', function() {
            closeTutorialWizard(true); // Don't mark as completed if skipped
        });
    }
    
    if (prevBtn) {
        prevBtn.addEventListener('click', prevTutorialStep);
    }
    
    if (nextBtn) {
        nextBtn.addEventListener('click', nextTutorialStep);
    }
    
    if (finishBtn) {
        finishBtn.addEventListener('click', finishTutorial);
    }
    
    // Close on outside click
    modal.addEventListener('click', function(e) {
        if (e.target === modal) {
            closeTutorialWizard(true); // Don't mark as completed if closed by clicking outside
        }
    });
}

function initializeAboutModal() {
    const aboutModal = document.getElementById('aboutModal');
    const closeAboutModalBtn = document.getElementById('closeAboutModal');
    
    if (closeAboutModalBtn) {
        closeAboutModalBtn.addEventListener('click', closeAboutModal);
    }
    
    // Close about modal when clicking outside
    if (aboutModal) {
        aboutModal.addEventListener('click', function(e) {
            if (e.target === aboutModal) {
                closeAboutModal();
            }
        });
    }
}
